// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Shadow.A.1.fsh                                 ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.0 (2017/03/23)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose : To simulate a shadow.                                        ==
// ==                                                                        ==
// == Description: The source image is distorted with a combination of shear ==
// == and taper transforms and a one dimensional blur is then applied. It is ==
// == expected that this shader will normaly be used to create a shadow of a ==
// == clipSprite, in which case color: and opacity: clauses would normaly be ==
// == used in the .scn file to get a black or dark grey partialy transparent ==
// == shadow. Similarly a scale: clause can be used to produce short or long ==
// == shadows.                                                               ==
// ==                                                                        ==
// == Note this version of the shader can produce a reasonable simulation of ==
// == the shadow of a clipSprite only when that shadow falls entirely on the ==
// == floor on which the performer depicted in the clipSprite is dancing. It ==
// == does not produce a realistic image if the shadow falls partialy on the ==
// == floor and partialy on a wall or other surface.                         ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/03/23 - v1.0.0 - Initial version.                               ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manually copying relevant utility code snippets from the sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) ( (xy)/(wsize) )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters. Note, a hotspot of (0.5,0.5)
// is required to produce a symmetric result.

uniform vec2 Emu_Shadow_scale;
uniform vec2 Emu_Shadow_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Shadow_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Shadow_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The shear and taper for X and Y are specified by the following parameters.
// Value of zero for an axis produces no distortion.  Values  are expected to
// be in the range -1.0 to +1.0, but values outside that range are valid.  It
// is expected that taper would normally only be used in a scene using  a  2D
// camera, where rotations about the X or Y axes do not work, in order to get
// the effect of a long shadow tapering into the distance. In 3D scenes it is
// more natural to rotate about the X axis to achieve this effect.

uniform vec2  Emu_Shadow_shear;
uniform vec2  Emu_Shadow_taper;

#define shear Emu_Shadow_shear
#define taper Emu_Shadow_taper

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The shadow's degree of blur is controlled by the following parameter which
// defaults to 0.0 corresponding to no blur.  Values are expected to be in the
// range 0.0 to 10.0 but values outside that range are valid.

uniform float Emu_Shadow_blur;
#define blur  Emu_Shadow_blur

// ============================================================================
// == Blur filter definition ==================================================
// ============================================================================

// The shadow is blurred using a simple one dimensional convolutional filter.
// The code used here was derived from that used for the more general filters
// in the TheEmuLib.Filters directory and their sources should  be  consulted
// if more information is wanted. A two dimensional blur could be used but it
// would not produce a significantly better result for a clipSprite shadow.

vec2 sample_step = vec2 ( blur/u_WindowSize.x, 1.0 );

#define BLUR QQ(-4.0,0.1) + QQ(-3.0,0.1) \
           + QQ(-2.0,0.1) + QQ(-1.0,0.1) \
                   + QQ(0.0,0.2)         \
           + QQ(+1.0,0.1) + QQ(+2.0,0.1) \
           + QQ(+3.0,0.1) + QQ(+4.0,0.1) \

#define CL(xy) clamp(xy,vec2(0.0),vec2(1.0))
#define TX(xy) texture2D ( iChannel0, CL(xy*sample_step+st) )

#define QQ(xx,ww) ( TX(vec2(xx,0.0)) * ww )

// ============================================================================
// == The shader's major functions ============================================
// ============================================================================

vec4 Emu_Shadow ( vec2 uv )
 {
   // Apply the transform with its fixed point at the hotspot.  Note, the code
   // here is the inverse of the required transform because we are determining
   // what point in the source image will be transformed to the current pixel.

   vec2 st = ( uv - hotspot ) / scale;

   st = st - shear*st.yx;
   st = st / ( 1.0 - taper*st.yx );

   st = st + hotspot;

   // Determine the colour using transparent black if the point
   // lies outside of the area of the transformed source image.

   return ( st == fract(abs(st)) ) ? (BLUR) : vec4(0.0);

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Get the normalised coordinates of the current point.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   // Apply the transform and update the shader's outputs.

   gl_FragColor = Emu_Shadow(uv) * gl_Color;

 }

// ============================================================================
